/**
 * \file: caam_develmode_loader.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: Special module loading binary to prevent startup in
 *       non-secure mode if the factory trigger is existing.
 *       In case the file is missing or access fails the device
 *       may be started in non-secure mode.
 *
 * \author: Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2016 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <syslog.h>
#include <sys/stat.h>
#include <ctype.h>
#include <libkmod.h>
#include <sdc.h>
#include <private/sdc_advanced.h>

#define FACTORY_TRIGGER_CFG "sdc_imx6_factory_trigger"
#define CAAM_MODULE_NAME "caam"
#define CAAM_MODULE_PARAM "ssm_st_ignore"
#define MODULE_PARAM_CHAR_LEN (strlen(CAAM_MODULE_PARAM)+1)

typedef enum caam_develmode_error_code_t {
    CDML_OK,
    CDML_MODULE_LOADED,
    CDML_CONF_MISSING,
    CDML_CONF_INVALID,
    CDML_TRIGGER_PRESENT,
    CDML_ERROR
} caam_develmode_error_code_t;

caam_develmode_error_code_t prepare_caam_module(struct kmod_ctx *kmodule_ctx, struct kmod_module **mod)
{
    int result;

    result = kmod_module_new_from_name(kmodule_ctx,CAAM_MODULE_NAME,mod);
    if (result) {
        syslog(LOG_ERR | LOG_DAEMON, "Failed to get module from name - error %d\n", result);
        return CDML_ERROR;
    }

    result = kmod_module_get_initstate(*mod);
    if (result == KMOD_MODULE_BUILTIN) {
        syslog(LOG_DEBUG| LOG_DAEMON, "Module %s is built-in\n", CAAM_MODULE_NAME);
        return CDML_MODULE_LOADED;
    }
    if (result > 0) {
        syslog(LOG_DEBUG | LOG_DAEMON, "Module %s is already loaded\n", CAAM_MODULE_NAME);
        return CDML_MODULE_LOADED;
    }
    /* < 0 module not in kernel */

    return CDML_OK;
}

caam_develmode_error_code_t insert_caam_module(struct kmod_module *mod, const char *param)
{
    int result;

    syslog(LOG_INFO | LOG_DAEMON, "Insert module %s params \"%s\"\n",
           CAAM_MODULE_NAME, param);
    result=kmod_module_insert_module(mod,0,param);
    //normally we are not trying to load something if it is there, we check it in advance.
    //But there might occur a case (which actually happened) that someone else (udevd)
    //loads our module while we are waiting for a dependency to be resolved. Actually, loading
    //the module we waited for, triggered udevd to load our module as well. One of both is faster ...
    if (result<0 && result!=-EEXIST) {
        syslog(LOG_ERR | LOG_DAEMON, "Failed to insert module - error %d\n", result);
        return CDML_ERROR;
    }
    if (result == -EEXIST) {
        syslog(LOG_INFO | LOG_DAEMON, "Module %s is already loaded\n", CAAM_MODULE_NAME);
    }

    return CDML_OK;
}

caam_develmode_error_code_t fill_param_str (char *param_str, int param_len)
{
    sdc_error_t sdc_err;
    caam_develmode_error_code_t cdml_err = CDML_OK;
    char *factory_trigger_path = NULL;
    int err;
    int len;

    /* initalize param str to no parameter */
    memset(param_str, '\0', param_len);

    /* read sdc config */
    sdc_err = sdc_config_file_lookup(&factory_trigger_path, FACTORY_TRIGGER_CFG);
    if (sdc_err == SDC_CONFIG_MISSING) {
        /* probe caam without any modul_param */
        cdml_err = CDML_CONF_MISSING;
    } else {
        if ((sdc_err != SDC_OK) || (!factory_trigger_path)) {
            syslog(LOG_ERR | LOG_DAEMON, "Error reading factory trigger config - %s\n",
                   sdc_get_error_string(sdc_err));
            cdml_err = CDML_CONF_INVALID;
        }
    }

    if (cdml_err == CDML_OK) {
        if (0 == access(factory_trigger_path, R_OK)) {
            syslog(LOG_INFO | LOG_DAEMON, "Trigger file present - disallow non-secure startup\n");
            cdml_err = CDML_TRIGGER_PRESENT;
        } else {
            err = errno;

            if (err != ENOENT) {
                syslog(LOG_ERR | LOG_DAEMON, "Unexpected error %s when trying to access %s\n",
                       strerror(err),
                       factory_trigger_path);
                cdml_err = CDML_ERROR;
            }
        }
    }

    if (factory_trigger_path)
        free(factory_trigger_path);

    if (cdml_err == CDML_OK) {
        len = snprintf(param_str, param_len, "%s", CAAM_MODULE_PARAM);
        if (len != strlen(CAAM_MODULE_PARAM)) {
            syslog(LOG_ERR | LOG_DAEMON, "Unexpected error writing module parameter\n");
            cdml_err = CDML_ERROR;
        }
    }

    return cdml_err;
}

int main(int argc, char **argv)
{
    struct kmod_ctx *kmodule_ctx;
    struct kmod_module *mod=NULL;
    caam_develmode_error_code_t err;
    char param_str[MODULE_PARAM_CHAR_LEN];

    (void)argv;

    if (argc != 1) {
        syslog(LOG_ERR | LOG_DAEMON, "Invalid arguments\n");
        return EXIT_FAILURE;
    }

    kmodule_ctx=kmod_new(NULL,NULL);
    if (!kmodule_ctx)
        return EXIT_FAILURE;

    err = prepare_caam_module(kmodule_ctx, &mod);

    if (err == CDML_OK) {
        err = fill_param_str(param_str, MODULE_PARAM_CHAR_LEN);

        /*
         * if err != CDML_ERROR - param_str will be initialized to ""
         * continue loading module without parameters
         */
        if (err != CDML_ERROR)
            err = insert_caam_module(mod, param_str);
    }

    if (mod)
        kmod_module_unref(mod);

    kmod_unref(kmodule_ctx);

    if (err == CDML_ERROR) {
        return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}

